<?php
/* --------------------------------------------------------------
   LinkPageContent.inc.php 2019-06-21
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

/**
 * Class LinkPageContent
 *
 * This class represents a link page content entity
 *
 * @category   System
 * @package    Content
 */
class LinkPageContent
    implements ContentAttributesInterface, ContentNameAttributeInterface, ContentLinkAttributesInterface, ContentGroupIdInterface
{
    /**
     * Content type
     *
     * @var string
     */
    protected $type;
    
    /**
     * Content position
     *
     * @var string
     */
    protected $position;
    
    /**
     * Content is active
     *
     * @var bool
     */
    protected $status;
    
    /**
     * Localized content titles
     *
     * @var ContentTitleCollection
     */
    protected $titles;
    
    
    /**
     * Localized content names
     *
     * @var ContentNameCollection
     */
    protected $names;
    
    /**
     * Localized content links
     *
     * @var ContentLinkCollection
     */
    protected $links;
    
    /**
     * content is opening in new tab
     *
     * @var bool
     */
    protected $opensInNewTabStatus;
    
    /**
     * Content group ID
     *
     * @var ContentIdentificationInterface
     */
    protected $id;
    
    
    /**
     * LinkPageContent constructor
     *
     * @param ContentPositionInterface  $pagePosition        Page position
     * @param ContentStatusInterface    $status              Activation status
     * @param ContentTitleCollection    $titles              Localized titles
     * @param ContentNameCollection     $names               Localized names
     * @param ContentLinkCollection     $links               Localized links
     * @param ContentOpenInNewTabStatus $opensInNewTabStatus Whether the link is opened in a new tab
     * @param ContentIdentificationInterface|null                  $id                  Optional content group ID
     */
    public function __construct(
        ContentPositionInterface $pagePosition,
        ContentStatusInterface $status,
        ContentTitleCollection $titles,
        ContentNameCollection $names,
        ContentLinkCollection $links,
        ContentOpenInNewTabStatus $opensInNewTabStatus,
        ContentIdentificationInterface $id = null
    ) {
        $this->type                = ContentType::createForContent()->type();
        $this->position            = $pagePosition->position();
        $this->status              = $status->isActive();
        $this->titles              = $titles;
        $this->names               = $names;
        $this->links               = $links;
        $this->opensInNewTabStatus = $opensInNewTabStatus->opensInNewTab();
        $this->id                  = $id ?? null;
    }
    
    
    /**
     * Return the content type
     *
     * @return string
     */
    public function type(): ?string
    {
        return $this->type;
    }
    
    
    /**
     * Return the content position
     *
     * @return string
     */
    public function position(): ?string
    {
        return $this->position;
    }
    
    
    /**
     * Return whether the content is active
     *
     * @return bool
     */
    public function isActive(): ?bool
    {
        return $this->status;
    }
    
    
    /**
     * Return the localized content titles
     *
     * @return ContentTitleCollection
     */
    public function titles(): ?ContentTitleCollection
    {
        return $this->titles;
    }
    
    
    /**
     * Return the localized content names
     *
     * @return ContentNameCollection
     */
    public function names(): ?ContentNameCollection
    {
        return $this->names;
    }
    
    
    /**
     * Return the localized content links
     *
     * @return ContentLinkCollection
     */
    public function links(): ?ContentLinkCollection
    {
        return $this->links;
    }
    
    
    /**
     * Return whether the content is opened in a new tab
     *
     * @return bool
     */
    public function isOpeningInNewTab(): ?bool
    {
        return $this->opensInNewTabStatus;
    }
    
    
    /**
     * Returns the content group id if set, otherwise null will be returned.
     *
     * @return ContentIdentificationInterface|null
     */
    public function id() : ?ContentIdentificationInterface
    {
        return $this->id;
    }
}